<?php
require("config.php");
session_start();

header("Content-Type: application/json");

// --- Validate session ---
if (!isset($_SESSION['SESS_USERID'])) {
    echo json_encode(["success" => false, "message" => "User not logged in"]);
    exit;
}

$userId = $_SESSION['SESS_USERID'];

// --- Helper function: simulate live rate ---
function getLiveMarketRate($symbol) {
    $mockRates = [
        "BTC/USD" => 101500.00,
        "ETH/USD" => 2350.00,
        "XAU/USD" => 1950.50
    ];
    return $mockRates[$symbol] ?? 1000.00;
}

// --- Collect and sanitize form data ---
$action   = trim($_POST['optionr'] ?? ''); // buy or sell
$pair     = trim($_POST['pair'] ?? '');
$amount   = floatval($_POST['amount'] ?? 0);
$tp       = floatval($_POST['tp'] ?? 0);
$sl       = floatval($_POST['sl'] ?? 0);
$leverage = intval($_POST['leverage'] ?? 1);
$duration = trim($_POST['duration'] ?? '1m');
$pip      = intval($_POST['pip'] ?? 0); // from pip selector if available
$email    = trim($_POST['email'] ?? '');

// --- Validate data ---
if ($pair === '' || $amount <= 0 || !in_array($action, ['buy', 'sell'])) {
    echo json_encode(["success" => false, "message" => "Invalid or missing trade data"]);
    exit;
}

// --- Validate pair exists in pairs table ---
$pairStmt = $pdo->prepare("SELECT * FROM pairs WHERE symbol = ? AND status = 1 LIMIT 1");
$pairStmt->execute([$pair]);
$pairRow = $pairStmt->fetch(PDO::FETCH_ASSOC);

if (!$pairRow) {
    echo json_encode(["success" => false, "message" => "Invalid or inactive trading pair"]);
    exit;
}

$entryRate = getLiveMarketRate($pair);
$userCurrency = $pairRow['quote'];
$assetType = $pairRow['type'] ?? '';

// --- Start transaction ---
$pdo->beginTransaction();

try {
    // --- Fetch user details ---
    $usr = $pdo->prepare("SELECT email, iprofit FROM logins WHERE id = ?");
    $usr->execute([$userId]);
    $user = $usr->fetch(PDO::FETCH_ASSOC);

    if (!$user) {
        throw new Exception("User not found");
    }

    $email = $user['email'];
    $currentProfit = floatval($user['iprofit']);

    // --- Simulated profit/loss outcome ---
    // You can replace this with real calculations later
    $userProfit = ($action === 'buy')
        ? round($amount * 0.02, 2)   // +2%
        : round($amount * -0.01, 2); // -1%

    $newTotalProfit = $currentProfit + $userProfit;

    // --- Insert trade record ---
    $insert = $pdo->prepare("
        INSERT INTO trade
        (log_id, email, system, tp, sl, leverage, plan, asset, signa, status, istatus, optionr,
         lot, invested, user_iprofit, user_bal, user_cur, Wrate, user_rate,
         entryrate, closerate, entrytime, expirytime, duratn, returnr, date)
        VALUES
        (:log_id, :email, 'livetrade', :tp, :sl, :leverage, '', :asset, '', 0, 0, :optionr,
         :lot, :invested, :user_iprofit, 0, :user_cur, 0, 0,
         :entryrate, 0, NOW(), NULL, :duratn, 0, CURDATE())
    ");

    $insert->execute([
        ':log_id'       => $userId,
        ':email'        => $email,
        ':tp'           => $tp,
        ':sl'           => $sl,
        ':leverage'     => $leverage,
        ':asset'        => $pair,
        ':optionr'      => $action,
        ':lot'          => $pip,
        ':invested'     => $amount,
        ':user_iprofit' => $userProfit,
        ':user_cur'     => $userCurrency,
        ':entryrate'    => $entryRate,
        ':duratn'       => $duration
    ]);

    // --- Update total user profit in logins table ---
    $pdo->prepare("UPDATE logins SET iprofit = ? WHERE id = ?")
        ->execute([$newTotalProfit, $userId]);

    // --- Calculate new Win Rate for closed trades ---
    $wrateStmt = $pdo->prepare("
        SELECT ROUND(
            (SUM(CASE WHEN user_iprofit > 0 THEN 1 ELSE 0 END) / COUNT(*)) * 100, 2
        ) AS winRate
        FROM trade
        WHERE log_id = ? AND system='livetrade' AND status=1
    ");
    $wrateStmt->execute([$userId]);
    $winRate = floatval($wrateStmt->fetchColumn() ?? 0);

    $pdo->prepare("UPDATE logins SET wrate = ? WHERE id = ?")
        ->execute([$winRate, $userId]);

    $pdo->commit();

    echo json_encode([
        "success" => true,
        "message" => "Trade executed successfully!",
        "pair" => $pair,
        "entryRate" => $entryRate,
        "profit" => $userProfit,
        "totalProfit" => $newTotalProfit,
        "winRate" => $winRate
    ]);

} catch (Exception $e) {
    $pdo->rollBack();
    echo json_encode(["success" => false, "message" => "Error: " . $e->getMessage()]);
}
?>
