<?php
// process_trades.php
require("../config.php"); // must set $pdo and session_start()
header('Content-Type: application/json');

if (!isset($_SESSION['SESS_USERID'])) {
    echo json_encode(['success' => false, 'msg' => 'Not authenticated']);
    exit;
}
$userId = $_SESSION['SESS_USERID'];

/**
 * Get market price for a pair.
 * This simple implementation uses Binance for crypto pairs (BTC/USD -> BTCUSDT).
 * Modify as needed for forex/commodities (other APIs).
 */
function getMarketPriceAPI($symbol) {
    // Normalize symbol -> Binance expects e.g. BTCUSDT or ETHUSDT
    // We'll convert common forms "BTC/USD" -> "BTCUSDT"
    $s = strtoupper($symbol);
    $s = str_replace('/', '', $s); // BTCUSD
    // if user passed USD, use USDT to map to Binance
    if (substr($s, -3) === 'USD') {
        $s = substr($s, 0, -3) . 'USDT';
    }
    // if symbol already ends with USDT or BTC etc, keep it.
    $url = "https://api.binance.com/api/v3/ticker/24hr?symbol=" . urlencode($s);
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 5);
    $res = curl_exec($ch);
    $err = curl_error($ch);
    curl_close($ch);
    if ($res === false || !$res) return ['ok' => false, 'msg' => $err ?: 'no response'];
    $json = json_decode($res, true);
    if (!$json || isset($json['code'])) {
        return ['ok' => false, 'msg' => 'API response error'];
    }
    // response contains lastPrice, highPrice, lowPrice, volume etc
    return [
        'ok' => true,
        'price' => (float)$json['lastPrice'],
        'high' => $json['highPrice'] ?? null,
        'low' => $json['lowPrice'] ?? null,
        'volume' => $json['volume'] ?? null
    ];
}

/**
 * Helper: compute expiry timestamp from duration string (e.g. "1m", "5m", "1h")
 */
function computeExpiry($duration) {
    $now = new DateTime('now', new DateTimeZone('UTC'));
    if (preg_match('/^(\d+)(m|h|d)$/', $duration, $m)) {
        $n = intval($m[1]);
        $unit = $m[2];
        if ($unit === 'm') $now->modify("+{$n} minutes");
        if ($unit === 'h') $now->modify("+{$n} hours");
        if ($unit === 'd') $now->modify("+{$n} days");
    } else {
        $now->modify("+1 minutes");
    }
    return $now->format('Y-m-d H:i:s');
}

/**
 * Close a trade (update DB): computes profit and updates iprofit
 * Parameters:
 *   $trade array row
 *   $closerate float
 */
function closeTrade(PDO $pdo, $trade, $closerate) {
    // compute profit with formula: percent change * leverage * invested
    $entry = (float)$trade['entryrate'];
    $invested = (float)$trade['invested'];
    $leverage = (float)$trade['leverage'];
    $option = strtolower($trade['optionr']);

    if ($entry == 0) $entry = 0.0000001; // avoid div by zero

    // percent change (close - entry) / entry
    if ($option === 'buy') {
        $percent = ($closerate - $entry) / $entry;
    } else {
        $percent = ($entry - $closerate) / $entry;
    }

    $profit = $invested * $percent * $leverage;
    $profit = round($profit, 2);

    // update trade row: set closerate, expirytime now, status=2, returnr=profit
    $stmt = $pdo->prepare("UPDATE trade SET closerate = ?, expirytime = ?, status = 2, returnr = ? WHERE id = ?");
    $stmt->execute([$closerate, date('Y-m-d H:i:s'), $profit, $trade['id']]);

    // update user's iprofit (increment)
    $stmt = $pdo->prepare("UPDATE logins SET iprofit = iprofit + ? WHERE id = ?");
    $stmt->execute([$profit, $trade['log_id']]);

    return $profit;
}

// Route actions
$action = $_REQUEST['action'] ?? '';

if ($action === 'fetch_price') {
    $symbol = $_GET['symbol'] ?? 'BTC/USD';
    $result = getMarketPriceAPI($symbol);
    if (!$result['ok']) {
        echo json_encode(['success' => false, 'msg' => $result['msg']]);
        exit;
    }
    echo json_encode([
        'success' => true,
        'price' => $result['price'],
        'price_text' => '$' . number_format($result['price'], 2),
        'high' => $result['high'],
        'low' => $result['low'],
        'volume' => $result['volume']
    ]);
    exit;
}

// Place an open trade (user clicked Buy/Sell). This stores trade as status = 1 (open)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'place_open_trade') {
    $pair = $_POST['pair'] ?? 'BTC/USD';
    $option = $_POST['option'] ?? 'buy';
    $amount = floatval($_POST['amount'] ?? 0);
    $tp = floatval($_POST['tp'] ?? 0);
    $sl = floatval($_POST['sl'] ?? 0);
    $duration = $_POST['duration'] ?? '1m';
    $leverage = floatval($_POST['leverage'] ?? 1);

    if ($amount <= 0) {
        echo json_encode(['success' => false, 'msg' => 'Invalid amount']);
        exit;
    }

    // get current market price for entry
    $priceRes = getMarketPriceAPI($pair);
    if (!$priceRes['ok']) {
        echo json_encode(['success' => false, 'msg' => 'Price fetch failed']);
        exit;
    }
    $entryPrice = $priceRes['price'];

    // expiry time computed
    $expiry = computeExpiry($duration);

    // insert trade as open (status = 1)
    $stmt = $pdo->prepare("INSERT INTO trade
        (log_id, signal_id, email, invest_id, plan, asset, signa, status, optionr, invested, user_cur, user_rate, entryrate, closerate, entrytime, expirytime, duratn, returnr, leverage, date, tp, sl)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
    $now = date('Y-m-d H:i:s');
    $stmt->execute([
        $userId, null, $_SESSION['email'], $userId,
        'live', $pair, 'no signal',
        1, // status => open/trading
        $option,
        $amount,
        'USD',
        1, // user_rate placeholder
        $entryPrice,
        0,
        $now,
        $expiry,
        $duration,
        0,
        $leverage,
        date('Y-m-d'),
        $tp, $sl
    ]);
    $tradeId = $pdo->lastInsertId();

    // Optionally deduct reserved funds or adjust user balance logic here (not implemented)
    // Return success
    echo json_encode(['success' => true, 'trade_id' => $tradeId, 'entryrate' => $entryPrice]);
    exit;
}

// Poll/process open trades: check open trades and close if needed
if ($action === 'poll' || $action === '') {
    // find all open trades (status = 1)
    $stmt = $pdo->query("SELECT * FROM trade WHERE status = 1");
    $openTrades = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $closedTrades = [];
    foreach ($openTrades as $trade) {
        // get current market price for this trade's asset
        $pair = $trade['asset'] ?? 'BTC/USD';
        $priceRes = getMarketPriceAPI($pair);
        if (!$priceRes['ok']) {
            // skip this trade for now
            continue;
        }
        $currentPrice = $priceRes['price'];
        $nowTs = new DateTime('now', new DateTimeZone('UTC'));
        $expiryTs = new DateTime($trade['expirytime'], new DateTimeZone('UTC'));

        // check TP/SL (if provided)
        $hitTP = false;
        $hitSL = false;
        $tp = (float)$trade['tp'];
        $sl = (float)$trade['sl'];
        $option = strtolower($trade['optionr']);

        if ($tp > 0) {
            if ($option === 'buy' && $currentPrice >= $tp) $hitTP = true;
            if ($option === 'sell' && $currentPrice <= $tp) $hitTP = true;
        }
        if ($sl > 0) {
            if ($option === 'buy' && $currentPrice <= $sl) $hitSL = true;
            if ($option === 'sell' && $currentPrice >= $sl) $hitSL = true;
        }

        // check expiry
        $expired = ($nowTs >= $expiryTs);

        if ($hitTP || $hitSL || $expired) {
            // close trade at current price
            $profit = closeTrade($pdo, $trade, $currentPrice); // this updates trade & iprofit
            $closedTrades[] = [
                'id' => $trade['id'],
                'pair' => $pair,
                'closed_at' => date('Y-m-d H:i:s'),
                'close_price' => $currentPrice,
                'profit' => $profit,
                'reason' => $hitTP ? 'TP' : ($hitSL ? 'SL' : 'TIME')
            ];
        }
    }

    // return updated stats for current user
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM trade WHERE log_id = ?");
    $stmt->execute([$userId]);
    $total = (int)$stmt->fetchColumn();

    $stmt = $pdo->prepare("SELECT COUNT(*) FROM trade WHERE log_id = ? AND status = 2 AND returnr > 0");
    $stmt->execute([$userId]);
    $wins = (int)$stmt->fetchColumn();

    $winRate = $total > 0 ? round(($wins / $total) * 100, 2) : 0;

    $stmt = $pdo->prepare("SELECT COALESCE(SUM(returnr),0) FROM trade WHERE log_id = ? AND status = 2");
    $stmt->execute([$userId]);
    $totalProfit = (float)$stmt->fetchColumn();

    // fetch user's iprofit for balance update
    $stmt = $pdo->prepare("SELECT iprofit FROM logins WHERE id = ?");
    $stmt->execute([$userId]);
    $iprofit = (float)$stmt->fetchColumn();

    echo json_encode([
        'success' => true,
        'closed' => $closedTrades,
        'winRate' => $winRate,
        'totalProfit' => round($totalProfit, 2),
        'new_balance' => $iprofit
    ]);
    exit;
}

// fallback
echo json_encode(['success' => false, 'msg' => 'Unknown action']);
exit;
