<?php
/**
 * UNIVERSAL LIVE MARKET DATA FETCHER
 * Fetches real-time price, volume, high, low, open, close, and market cap
 * for ANY asset selected from your database (e.g. BTC/USD, SOL/USD, etc.)
 * using CoinGecko API.
 */
header('Content-Type: application/json');

$pair = strtoupper(trim($_GET['pair'] ?? ''));

if (!$pair) {
    echo json_encode(['success' => false, 'message' => 'No pair selected']);
    exit;
}

// Extract base coin (e.g. BTC from BTC/USD)
$parts = explode('/', $pair);
$base = strtolower(trim($parts[0] ?? ''));
if (!$base) {
    echo json_encode(['success' => false, 'message' => 'Invalid pair format']);
    exit;
}

// Fetch CoinGecko coins list to dynamically match the ID
$coinsListUrl = "https://api.coingecko.com/api/v3/coins/list";
$coinsListResponse = @file_get_contents($coinsListUrl);

if (!$coinsListResponse) {
    echo json_encode(['success' => false, 'message' => 'Failed to fetch coins list']);
    exit;
}

$coinsList = json_decode($coinsListResponse, true);

// Try to match user-selected asset symbol to a CoinGecko ID
$coinId = null;
foreach ($coinsList as $coin) {
    if (strtolower($coin['symbol']) === strtolower($base)) {
        $coinId = $coin['id'];
        break;
    }
}

if (!$coinId) {
    echo json_encode(['success' => false, 'message' => "Asset '$base' not found on CoinGecko"]);
    exit;
}

// Fetch live data for that coin
$url = "https://api.coingecko.com/api/v3/coins/{$coinId}?localization=false&tickers=false&market_data=true";
$response = @file_get_contents($url);

if (!$response) {
    echo json_encode(['success' => false, 'message' => 'Failed to fetch market data']);
    exit;
}

$data = json_decode($response, true);
if (!$data || !isset($data['market_data'])) {
    echo json_encode(['success' => false, 'message' => 'Invalid API response']);
    exit;
}

$market = $data['market_data'];

// Prepare and format response
$result = [
    'success'    => true,
    'cur_symbol' => '$',
    'symbol'     => strtoupper($pair),
    'open'       => round(($market['current_price']['usd'] ?? 0) - ($market['price_change_24h_in_currency']['usd'] ?? 0), 2),
    'high'       => round($market['high_24h']['usd'] ?? 0, 2),
    'low'        => round($market['low_24h']['usd'] ?? 0, 2),
    'close'      => round($market['current_price']['usd'] ?? 0, 2),
    'volume'     => round($market['total_volume']['usd'] ?? 0, 2),
    'marketCap'  => round($market['market_cap']['usd'] ?? 0, 2),
];

// Output JSON
echo json_encode($result);
