<?php
require("header.php"); // includes ../config.php + session

// --- TRADE OPEN LOGIC ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'openTrade') {
    header('Content-Type: application/json');
    try {
        if (!isset($_SESSION['SESS_USERID'])) throw new Exception("Session expired.");

        $userId = $_SESSION['SESS_USERID'];
        $pair = trim($_POST['pair'] ?? '');
        $tp = floatval($_POST['tp'] ?? 0);
        $sl = floatval($_POST['sl'] ?? 0);
        $amount = floatval($_POST['amount'] ?? 0);
        $optionr = strtolower($_POST['optionr'] ?? '');
        $duration = $_POST['duration'] ?? '1m';
        $leverage = intval($_POST['leverage'] ?? 100);

        if (!$pair || !in_array($optionr, ['buy', 'sell']) || $amount <= 0)
            throw new Exception("Invalid trade input.");

        $stmt = $pdo->prepare("SELECT * FROM logins WHERE id=?");
        $stmt->execute([$userId]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$user) throw new Exception("User not found.");

        if ($user['balance'] < $amount) throw new Exception("Insufficient balance.");

        $pairStmt = $pdo->prepare("SELECT * FROM pairs WHERE symbol=?");
        $pairStmt->execute([$pair]);
        $pairData = $pairStmt->fetch(PDO::FETCH_ASSOC);
        if (!$pairData) throw new Exception("Invalid trading pair.");

        $entryrate = getLiveMarketRate($pair);
        $entrytime = date('Y-m-d H:i:s');

        // Insert trade
        $insert = $pdo->prepare("
            INSERT INTO trade (
                log_id, signal_id, email, invest_id, system, tp, sl, plan, asset, signa,
                status, optionr, invested, user_cur, user_rate, entryrate, entrytime, leverage, date
            ) VALUES (
                :log_id, NULL, :email, :invest_id, 'livetrade', :tp, :sl, :plan, :asset, 'nosignal',
                1, :optionr, :invested, :user_cur, :user_rate, :entryrate, :entrytime, :leverage, NOW()
            )
        ");
        $insert->execute([
            'log_id' => $userId,
            'email' => $user['email'],
            'invest_id' => $user['invest_id'],
            'tp' => $tp,
            'sl' => $sl,
            'plan' => $pairData['type'],
            'asset' => $pairData['symbol'],
            'optionr' => $optionr,
            'invested' => $amount,
            'user_cur' => $user['cur_shot_name'],
            'user_rate' => $user['cur_rate'],
            'entryrate' => $entryrate,
            'entrytime' => $entrytime,
            'leverage' => $leverage
        ]);

        // Deduct balance
        $pdo->prepare("UPDATE logins SET balance = balance - :amt WHERE id = :id")
            ->execute(['amt' => $amount, 'id' => $userId]);

        echo json_encode(['success' => true, 'message' => 'Trade started successfully!']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
    exit;
}

// --- PRICE FETCH ENDPOINT (for AJAX refresh) ---
if (isset($_GET['pair'])) {
    header('Content-Type: application/json');
    echo json_encode([
        'price' => getLiveMarketRate($_GET['pair']),
        'timestamp' => date('H:i:s')
    ]);
    exit;
}

// --- SIMULATED LIVE RATE FUNCTION ---
function getLiveMarketRate($symbol) {
    $mock = [
        'BTC/USD' => 101500 + rand(-300, 300),
        'ETH/USD' => 2350 + rand(-10, 10),
        'XAU/USD' => 1950 + rand(-5, 5),
    ];
    return $mock[$symbol] ?? 1000 + rand(-20, 20);
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<title>Live Trading</title>
<script src="https://s3.tradingview.com/tv.js"></script>
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
<style>
body { background-color:#0F0A1F; color:white; font-family:sans-serif; }
.container { display:grid; grid-template-columns: 2fr 1fr; gap:20px; padding:20px; }
.card { background:linear-gradient(to bottom right,#1A1428,#0F0A1F); border:1px solid #333; border-radius:12px; padding:15px; }
.btn { padding:10px; border-radius:8px; font-weight:bold; cursor:pointer; width:100%; }
.btn-buy { background:linear-gradient(to right,#16a34a,#22c55e); color:black; }
.btn-sell { background:linear-gradient(to right,#dc2626,#ef4444); color:black; }
select, input { width:100%; padding:8px; border-radius:6px; background:#1A1428; border:1px solid #333; color:white; }
</style>
</head>
<body>

<div class="container">
  <!-- LEFT SIDE: TRADINGVIEW CHART -->
  <div class="card">
    <h3><i class="fa-solid fa-chart-line text-[#2FE6DE]"></i> Live Market Chart</h3>
    <div id="tradingview_chart" style="height:600px;"></div>

    <div id="livePrice" class="mt-3 text-lg">
      <strong>Current Price:</strong> <span id="currentPrice">Loading...</span>
      <small class="text-gray-400" id="priceTime"></small>
    </div>
  </div>

  <!-- RIGHT SIDE: TRADE FORM -->
  <div class="card">
    <h3><i class="fa-solid fa-wallet text-[#2FE6DE]"></i> Start a Live Trade</h3>
    <form id="tradeForm" method="POST">
      <input type="hidden" name="action" value="openTrade">

      <label>Trading Pair</label>
      <select name="pair" id="pair" required>
        <option value="">Select Pair</option>
        <?php
        $pairs = $pdo->query("SELECT * FROM pairs WHERE status=1 ORDER BY symbol")->fetchAll(PDO::FETCH_ASSOC);
        foreach ($pairs as $p) {
          echo '<option value="'.htmlspecialchars($p['symbol']).'">'.htmlspecialchars($p['symbol']).' ('.$p['type'].')</option>';
        }
        ?>
      </select>

      <div style="display:flex; gap:10px; margin-top:10px;">
        <input type="number" name="tp" id="tp" placeholder="Take Profit">
        <input type="number" name="sl" id="sl" placeholder="Stop Loss">
      </div>

      <label style="margin-top:10px;">Amount</label>
      <input type="number" name="amount" id="amount" min="1" placeholder="Enter amount">

      <label style="margin-top:10px;">Duration</label>
      <select name="duration" id="duration">
        <option value="1m">1m</option><option value="5m">5m</option>
        <option value="15m">15m</option><option value="1h">1h</option>
        <option value="4h">4h</option><option value="1d">1d</option>
      </select>

      <div style="display:flex; gap:10px; margin-top:15px;">
        <button type="button" id="buyTrade" data-option="buy" class="btn btn-buy">Buy</button>
        <button type="button" id="sellTrade" data-option="sell" class="btn btn-sell">Sell</button>
      </div>
      <div id="tradeMessage" style="margin-top:15px; font-size:0.9rem;"></div>
    </form>
  </div>
</div>

<script>
// --- TradingView Chart Loader ---
let tvWidget;
function loadTradingViewChart(symbol="BTCUSD") {
    if (tvWidget) tvWidget.remove();
    tvWidget = new TradingView.widget({
        autosize: true,
        symbol: symbol,
        interval: "1",
        timezone: "Etc/UTC",
        theme: "dark",
        style: "1",
        locale: "en",
        toolbar_bg: "#0F0A1F",
        enable_publishing: false,
        allow_symbol_change: false,
        container_id: "tradingview_chart"
    });
}

// --- Auto Refresh Current Price ---
function refreshPrice() {
    const pair = document.getElementById('pair').value;
    if (!pair) return;
    fetch('live_trade.php?pair=' + encodeURIComponent(pair))
      .then(r => r.json())
      .then(d => {
          document.getElementById('currentPrice').textContent = d.price.toFixed(2);
          document.getElementById('priceTime').textContent = '(' + d.timestamp + ')';
      });
}
setInterval(refreshPrice, 5000); // refresh every 5s

// --- Load chart when pair changes ---
document.getElementById('pair').addEventListener('change', e => {
    const symbol = e.target.value.replace('/', '');
    loadTradingViewChart(symbol);
    refreshPrice();
});

// --- Submit Trade (Buy/Sell) ---
document.querySelectorAll('#buyTrade,#sellTrade').forEach(btn=>{
  btn.addEventListener('click',function(){
    const form=document.getElementById('tradeForm');
    const data=new FormData(form);
    data.append('optionr',this.dataset.option);
    const msg=document.getElementById('tradeMessage');
    msg.innerHTML="<span style='color:yellow'>Processing...</span>";
    fetch('live_trade.php',{method:'POST',body:data})
    .then(r=>r.json())
    .then(res=>{
      msg.innerHTML = res.success
        ? "<span style='color:lime'>" + res.message + "</span>"
        : "<span style='color:red'>" + res.message + "</span>";
    })
    .catch(err=> msg.innerHTML="<span style='color:red'>Error: "+err+"</span>");
  });
});

// Load default BTC chart
loadTradingViewChart("BTCUSD");
refreshPrice();
</script>
</body>
</html>


        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        
        <div class="mt-3 md:mt-4 bg-gradient-to-br from-[#1A1428] to-[#0F0A1F] rounded-xl overflow-hidden shadow-xl border border-gray-800/30">
            <div class="p-3 md:p-4 border-b border-gray-800/30 flex items-center justify-between">
              
              
              
              
        
              
              
                <h2 class="text-base md:text-lg font-medium flex items-center gap-2">
                    <i class="fas fa-history text-[#2FE6DE] text-sm md:text-base"></i>
                    Recent Trades
                </h2>
                <a href="" class="text-[#2FE6DE] text-xs md:text-sm hover:underline flex items-center gap-1 px-2 md:px-3 py-1 rounded-lg bg-[#0F0A1F]/50 border border-gray-800/30">
                    View All
                    <i class="fas fa-arrow-right text-xs"></i>
                </a>
            </div>

            
          <?php
// Fetch last 8 trades for this user
$stmt = $pdo->prepare("SELECT * FROM trade WHERE log_id = :log_id ORDER BY id DESC LIMIT 1000");
$stmt->execute(['log_id' => $userId]);
$trades = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<div class="bg-[#0F0A1F] rounded-xl p-6 border border-gray-800/30 shadow-lg">

    <!-- Desktop View -->
    <div class="hidden md:block overflow-x-auto scrollbar-hide">
        <table class="w-full min-w-[900px]">
            <thead>
                <tr class="text-left text-gray-400 text-sm border-b border-gray-800/50">
                    <th class="pb-4">Trading Pair</th>
                    <th class="pb-4">Trading Type</th>
                    <th class="pb-4">Amount</th>
                    <th class="pb-4">Direction</th>
                    <th class="pb-4">Entry/Close</th>
                    <th class="pb-4">Time</th>
                    <th class="pb-4">Status</th>
                    <th class="pb-4">Result</th>
                    <th class="pb-4">Profit/Loss</th>
                    <th class="pb-4 text-right pr-2">Details</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($trades as $trade): 
                    $direction = strtolower($trade['optionr']) === 'buy' ? 'Buy' : 'Sell';
                    $directionColor = $direction === 'Buy' ? 'green-400' : 'red-400';
                    $statusText = $trade['status'] == 1 ? 'Pending' : 'Completed';
                    $statusColor = $trade['status'] == 1 ? 'yellow-400' : 'green-400';

                    // Result logic based on returnr
                    $result = ($trade['returnr'] > 1) ? 'Win' : 'Trading..';
                    $resultColor = ($trade['returnr'] > 1) ? 'green-400' : 'red-400';

                    // Profit/Loss shows returnr
                    $profitLoss = (float)$trade['returnr'];
                    $wrate = $trade['Wrate'] ?? 0;
                ?>
                <tr class="border-b border-gray-800/30 text-white hover:bg-[#191023]/50 transition-colors">
                    <!--<td class="py-4 pl-2">-->
                    <!--    <div class="flex items-center">-->
                    <!--        <div class="w-8 h-8 rounded-full bg-[#2A243B] flex items-center justify-center mr-2">-->
                    <!--            <i class="fas fa-robot text-[#2FE6DE]"></i>-->
                    <!--        </div>-->
                    <!--        <span class="font-medium"><?= htmlspecialchars($trade['signa']) ?></span>-->
                    <!--    </div>-->
                    <!--</td>-->
                    <td class="py-4"><?= htmlspecialchars($trade['asset']) ?></td>
                    <td class="py-4"><?= htmlspecialchars($trade['plan']) ?></td>
                    <td class="py-4"><?= htmlspecialchars($trade['user_cur']) ?><?= number_format($trade['invested'], 2) ?></td>
                    <td class="py-4">
                        <span class="px-2 py-1 rounded-full text-xs bg-<?= $directionColor ?>/20 text-<?= $directionColor ?>">
                            <i class="fas fa-arrow-<?= $direction === 'Buy' ? 'up' : 'down' ?> mr-1"></i>
                            <?= $direction ?>
                        </span>
                    </td>
                    <td class="py-4">
                        <div class="text-sm"><?= $trade['user_cur'] ?><?= number_format($trade['entryrate'], 2) ?></div>
                        <div class="text-xs text-green-400"><?= $trade['user_cur'] ?><?= number_format($trade['closerate'], 2) ?></div>
                    </td>
                    <td class="py-4">
                        <div class="text-sm"><?= date("M d, Y H:i", strtotime($trade['entrytime'])) ?></div>
                        <div class="text-xs text-gray-400"><?= date("M d, Y H:i", strtotime($trade['expirytime'])) ?></div>
                    </td>
                    <td class="py-4">
                        <span class="px-2 py-1 rounded-full text-xs bg-<?= $statusColor ?>/20 text-<?= $statusColor ?>">
                            <i class="fas fa-check-circle mr-1"></i> <?= $statusText ?>
                        </span>
                    </td>
                    <td class="py-4">
                        <span class="px-2 py-1 rounded-full text-xs bg-<?= $resultColor ?>/20 text-<?= $resultColor ?>">
                            <i class="fas fa-trophy mr-1"></i> <?= $result ?>
                        </span>
                    </td>
                    <td class="py-4">
                        <?php if ($trade['status'] == 1): ?>
                            <span class="text-yellow-400 font-medium">-- --</span>
                        <?php else: ?>
                            <span class="text-<?= $resultColor ?> font-medium">
                                <?= $trade['user_cur'] ?><?= number_format($profitLoss, 2) ?>
                                <span class="text-xs opacity-75">(<?= number_format($wrate, 2) ?>%)</span>
                            </span>
                        <?php endif; ?>
                    </td>
                    <td class="py-4 text-right pr-2">
                        <a href="trade_details.php?id=<?= $trade['id'] ?>" class="px-3 py-1.5 bg-[#2A243B] text-[#2FE6DE] rounded-lg hover:bg-[#382f50] transition-colors inline-flex items-center">
                            <i class="fas fa-eye mr-1.5"></i> Details
                        </a>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>

   <!-- Mobile horizontal scrollable table -->
<div id="scrollableTable" class="md:hidden overflow-x-auto scrollbar-thick scrollbar-thumb-[#2FE6DE] scrollbar-track-[#0F0A1F] rounded-lg border border-gray-800/30">
    <table class="min-w-[800px] w-full text-white text-sm border-collapse">
        <thead>
            <tr class="bg-[#191023] border-b border-gray-800/50 text-left">
                <th class="px-4 py-2 whitespace-nowrap">Type</th>
                <th class="px-4 py-2 whitespace-nowrap">Pair</th>
                <th class="px-4 py-2 whitespace-nowrap">Amount</th>
                <th class="px-4 py-2 whitespace-nowrap">Direction</th>
                <th class="px-4 py-2 whitespace-nowrap">Entry Price</th>
                <th class="px-4 py-2 whitespace-nowrap">Close Price</th>
                <th class="px-4 py-2 whitespace-nowrap">Time</th>
                <th class="px-4 py-2 whitespace-nowrap">Duration</th>
                <th class="px-4 py-2 whitespace-nowrap">Status</th>
                <th class="px-4 py-2 whitespace-nowrap">Result</th>
                <th class="px-4 py-2 whitespace-nowrap">Profit/Loss</th>
                <th class="px-4 py-2 whitespace-nowrap">Details</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($trades as $trade):
                $direction = strtolower($trade['optionr']) === 'buy' ? 'Buy' : 'Sell';
                $directionColor = $direction === 'Buy' ? 'green-400' : 'red-400';
                $statusText = $trade['status'] == 1 ? 'Pending' : 'Completed';
                $statusColor = $trade['status'] == 1 ? 'yellow-400' : 'green-400';

                $result = ($trade['returnr'] > 1) ? 'Win' : 'Loss';
                $resultColor = ($trade['returnr'] > 1) ? 'green-400' : 'red-400';

                $profitLoss = (float)$trade['returnr'];
                $wrate = $trade['Wrate'] ?? 0;

                // Duration calculation
                $entryTime = new DateTime($trade['entrytime']);
                $expiryTime = new DateTime($trade['expirytime']);
                $interval = $entryTime->diff($expiryTime);
                $durationStr = '';
                if ($interval->d > 0) $durationStr .= $interval->d . 'd ';
                if ($interval->h > 0) $durationStr .= $interval->h . 'h ';
                if ($interval->i > 0) $durationStr .= $interval->i . 'm';
                $durationStr = trim($durationStr);
                if ($durationStr === '') $durationStr = '0m';
            ?>
            <tr class="border-b border-gray-800/30 hover:bg-[#191023]/60 transition-colors">
                <!--<td class="px-4 py-2 whitespace-nowrap flex items-center">-->
                <!--    <div class="w-6 h-6 rounded-full bg-[#2A243B] flex items-center justify-center mr-2">-->
                <!--        <i class="fas fa-robot text-[#2FE6DE]"></i>-->
                <!--    </div>-->
                <!--    <?= htmlspecialchars($trade['signa']) ?>-->
                <!--</td>-->
                 <td class="px-4 py-2 whitespace-nowrap"><?= htmlspecialchars($trade['plan']) ?></td>
                 <td class="px-4 py-2 whitespace-nowrap"><?= htmlspecialchars($trade['asset']) ?></td>
                <td class="px-4 py-2 whitespace-nowrap"><?= htmlspecialchars($trade['user_cur']) ?><?= number_format($trade['invested'], 2) ?></td>
                <td class="px-4 py-2 whitespace-nowrap">
                    <span class="px-2 py-0.5 rounded-full text-xs bg-<?= $directionColor ?>/20 text-<?= $directionColor ?>">
                        <i class="fas fa-arrow-<?= $direction === 'Buy' ? 'up' : 'down' ?> mr-1"></i> <?= $direction ?>
                    </span>
                </td>
                <td class="px-4 py-2 whitespace-nowrap"><?= $trade['user_cur'] ?><?= number_format($trade['entryrate'], 2) ?></td>
                <td class="px-4 py-2 whitespace-nowrap text-green-400"><?= $trade['user_cur'] ?><?= number_format($trade['closerate'], 2) ?></td>
                <td class="px-4 py-2 whitespace-nowrap"><?= date("M d, Y H:i", strtotime($trade['entrytime'])) ?></td>
                <td class="px-4 py-2 whitespace-nowrap"><?= $durationStr ?></td>
                <td class="px-4 py-2 whitespace-nowrap">
                    <span class="px-2 py-0.5 rounded-full text-xs bg-<?= $statusColor ?>/20 text-<?= $statusColor ?>">
                        <?= $statusText ?>
                    </span>
                </td>
                <td class="px-4 py-2 whitespace-nowrap">
                    <span class="px-2 py-0.5 rounded-full text-xs bg-<?= $resultColor ?>/20 text-<?= $resultColor ?>">
                        <?= $result ?>
                    </span>
                </td>
                <td class="px-4 py-2 whitespace-nowrap font-medium">
                    <?php if ($trade['status'] == 1): ?>
                        <span class="text-yellow-400">-- --</span>
                    <?php else: ?>
                        <span class="text-<?= $resultColor ?>">
                            <?= $trade['user_cur'] ?><?= number_format($profitLoss, 2) ?>
                            <span class="text-xs opacity-75">(<?= number_format($wrate, 2) ?>%)</span>
                        </span>
                    <?php endif; ?>
                </td>
                <td class="px-4 py-2 whitespace-nowrap">
                    <a href="trade_details.php?id=<?= $trade['id'] ?>" class="px-3 py-1 bg-[#2A243B] text-[#2FE6DE] rounded-lg hover:bg-[#382f50] transition-colors inline-flex items-center">
                        <i class="fas fa-eye mr-1"></i> Details
                    </a>
                </td>
            </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
</div>

<script>
  document.addEventListener("DOMContentLoaded", function() {
    const container = document.getElementById('scrollableTable');
    if (!container) return;

    let direction = 1; // 1 = right, -1 = left
    const speed = 0.5; // pixels per frame
    let animationId;

    function scrollStep() {
      container.scrollLeft += direction * speed;

      // Reverse direction when reaching scroll ends
      if (container.scrollLeft + container.clientWidth >= container.scrollWidth) {
        direction = -1;
      } else if (container.scrollLeft <= 0) {
        direction = 1;
      }

      animationId = requestAnimationFrame(scrollStep);
    }

    // Intersection Observer to start scrolling only when visible
    const observer = new IntersectionObserver((entries) => {
      entries.forEach(entry => {
        if (entry.isIntersecting) {
          if (!animationId) animationId = requestAnimationFrame(scrollStep);
        } else {
          if (animationId) {
            cancelAnimationFrame(animationId);
            animationId = null;
          }
        }
      });
    }, { threshold: 0.1 });

    observer.observe(container);
  });
</script>
</div>
        </div>
    </div>
</div>















<div id="errorModal" class="fixed inset-0 z-50 overflow-y-auto hidden" aria-labelledby="modal-title" role="dialog" aria-modal="true">
    <div class="flex items-end justify-center min-h-screen pt-4 px-4 pb-20 text-center sm:block sm:p-0">
        <div class="fixed inset-0 bg-gray-900 bg-opacity-75 transition-opacity backdrop-blur-sm" aria-hidden="true"></div>
        <span class="hidden sm:inline-block sm:align-middle sm:h-screen" aria-hidden="true">&#8203;</span>
        <div class="inline-block align-bottom bg-[#0F0A1F] rounded-lg px-4 pt-5 pb-4 text-left overflow-hidden shadow-xl transform transition-all sm:my-8 sm:align-middle sm:max-w-lg sm:w-full sm:p-4 border border-gray-800/50">
            <div class="sm:flex sm:items-start">
                <div class="mx-auto flex-shrink-0 flex items-center justify-center h-12 w-12 rounded-full bg-red-100 sm:mx-0 sm:h-10 sm:w-10">
                    <i class="fas fa-exclamation-triangle text-red-600"></i>
                </div>
                <div class="mt-3 text-center sm:mt-0 sm:ml-4 sm:text-left">
                    <h3 class="text-lg leading-6 font-medium" id="modal-title">Error</h3>
                    <div class="mt-2">
                        <p class="text-sm text-gray-400" id="errorMessage"></p>
                    </div>
                </div>
            </div>
            <div class="mt-5 sm:mt-4 sm:flex sm:flex-row-reverse">
                <button type="button" class="mt-3 w-full inline-flex justify-center rounded-md border border-transparent shadow-sm px-4 py-2 bg-[#2FE6DE] text-base font-medium text-black hover:brightness-110 focus:outline-none sm:mt-0 sm:w-auto sm:text-sm" onclick="closeErrorModal()">
                    Close
                </button>
            </div>
        </div>
    </div>
</div>


<style>
.scrollbar-hide::-webkit-scrollbar {
    display: none;
}
.scrollbar-hide {
    -ms-overflow-style: none;
    scrollbar-width: none;
}
.scale-95 {
    transform: scale(0.95);
}
.scale-100 {
    transform: scale(1);
}
.opacity-0 {
    opacity: 0;
}
.opacity-100 {
    opacity: 1;
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const currentPairDropdown = document.getElementById('currentPairDropdown');
    const pairSelectorModal = document.getElementById('pairSelectorModal');
    const closePairSelector = document.getElementById('closePairSelector');
    const pairSearchInput = document.getElementById('pairSearchInput');
    const pairList = document.getElementById('pairList');
    const pairOptions = document.querySelectorAll('.pair-option');

    // Open pair selector
    currentPairDropdown.addEventListener('click', () => {
        pairSelectorModal.classList.remove('hidden');
        pairSelectorModal.classList.add('flex');
        
        // Animate modal in
        setTimeout(() => {
            const modalContent = document.getElementById('modalContent');
            modalContent.classList.remove('scale-95', 'opacity-0');
            modalContent.classList.add('scale-100', 'opacity-100');
            pairSearchInput.focus();
        }, 10);
    });

    // Close pair selector
    closePairSelector.addEventListener('click', () => {
        closeModal();
    });

    // Close modal when clicking outside
    pairSelectorModal.addEventListener('click', (e) => {
        if (e.target === pairSelectorModal) {
            closeModal();
        }
    });

    // Function to close modal with animation
    function closeModal() {
        const modalContent = document.getElementById('modalContent');
        modalContent.classList.remove('scale-100', 'opacity-100');
        modalContent.classList.add('scale-95', 'opacity-0');
        
        setTimeout(() => {
            pairSelectorModal.classList.add('hidden');
            pairSelectorModal.classList.remove('flex');
        }, 200);
    }

    // Search functionality
    pairSearchInput.addEventListener('input', (e) => {
        const searchTerm = e.target.value.toLowerCase();
        pairOptions.forEach(option => {
            const pairName = option.textContent.toLowerCase();
            option.style.display = pairName.includes(searchTerm) ? 'flex' : 'none';
        });
    });

    // Select pair
   pairOptions.forEach(option => {
    option.addEventListener('click', () => {
        const selectedPair = option.dataset.pair.replace('/', '');
        
        window.location.href = `/user/trading/trade/${selectedPair}`;
    });
});
});

document.addEventListener('DOMContentLoaded', function() {
    // Trading state
    const state = {
        tradeType: null,
        amount: 0,
        duration: '1m',
        symbol: 'BTCUSDT', // From controller
        currentPrice: 114339.99,
        profitPercentage: 80.00,
        ws: null,
        reconnectAttempts: 0,
        maxReconnectAttempts: 5,
        timeframe: '1' // Default timeframe added
    };

    // DOM Elements
    const elements = {
        currentPrice: document.getElementById('currentPrice'),
        priceChange: document.getElementById('priceChange'),
        volume: document.getElementById('volume'),
        high: document.getElementById('high'),
        low: document.getElementById('low'),
        tradeAmount: document.getElementById('tradeAmount'),
        walletBalance: document.getElementById('walletBalance'),
        riseButton: document.getElementById('riseButton'),
        fallButton: document.getElementById('fallButton'),
        placeTrade: document.getElementById('placeTrade'),
        tradeButtonText: document.getElementById('tradeButtonText'),
        tradeButtonLoader: document.getElementById('tradeButtonLoader'),
        potentialReturn: document.getElementById('potentialReturn'),
        amountError: document.getElementById('amountError'),
        mainPrice: document.getElementById('main-price'),
        mainChange: document.getElementById('main-change'),
        timeframeButtons: document.querySelectorAll('.timeframe-btn')
    };

    // Initialize TradingView Chart Widget Variable
    let tradingViewWidget;

    // Initialize TradingView Chart with default timeframe
    function initTradingView(timeframe = '1') {
        if (tradingViewWidget) {
            tradingViewWidget.remove();
        }

        // Convert timeframe to TradingView format
        const interval = getTradeViewInterval(timeframe);

        tradingViewWidget = new TradingView.widget({
            "container_id": "trading_chart",
            "autosize": true,
            "symbol": `HUOBI:${state.symbol}`,
            "interval": interval,
            "timezone": "Etc/UTC",
            "theme": "dark",
            "style": "1",
            "locale": "en",
            "toolbar_bg": "#1A1428",
            "enable_publishing": false,
            "hide_side_toolbar": true,
            "allow_symbol_change": false,
            "save_image": false,
            "backgroundColor": "#0F0A1F",
            "gridColor": "#222222",
            "hide_volume": false,
            "enabled_features": ["use_localstorage_for_settings"],
            "disabled_features": [
                "header_symbol_search",
                "symbol_search_hot_key",
                "header_compare",
                "header_settings",
                "header_chart_type",
                "header_screenshot",
                "header_saveload"
            ],
            "overrides": {
                "mainSeriesProperties.candleStyle.upColor": "#26a69a",
                "mainSeriesProperties.candleStyle.downColor": "#ef5350",
                "mainSeriesProperties.candleStyle.borderUpColor": "#26a69a",
                "mainSeriesProperties.candleStyle.borderDownColor": "#ef5350",
                "mainSeriesProperties.candleStyle.wickUpColor": "#26a69a",
                "mainSeriesProperties.candleStyle.wickDownColor": "#ef5350",
                "paneProperties.background": "#0F0A1F",
                "paneProperties.vertGridProperties.color": "#222222",
                "paneProperties.horzGridProperties.color": "#222222",
                "scalesProperties.textColor": "#999999"
            }
        });
    }

    // Convert our timeframe format to TradingView format
    function getTradeViewInterval(timeframe) {
        const map = {
            '1': '1',
            '5': '5',
            '15': '15',
            '30': '30',
            '60': '60',
            '240': '240',
            '1D': '1D'
        };
        return map[timeframe] || '1';
    }

    // Handle timeframe change
    function changeTimeframe(timeframe) {
        // Update button styles
        document.querySelectorAll('.timeframe-btn').forEach(btn => {
            btn.classList.remove('bg-[#2FE6DE]/10', 'text-[#2FE6DE]', 'border-[#2FE6DE]/30');
            btn.classList.add('text-gray-400', 'hover:text-white', 'border-gray-800/30', 'hover:border-gray-600');

            if (btn.dataset.timeframe === timeframe) {
                btn.classList.remove('text-gray-400', 'hover:text-white', 'border-gray-800/30', 'hover:border-gray-600');
                btn.classList.add('bg-[#2FE6DE]/10', 'text-[#2FE6DE]', 'border-[#2FE6DE]/30');
            }
        });

        // Update chart
        initTradingView(timeframe);
    }

    // Add to your event handlers
    function handleTimeframeChange(newTimeframe) {
        state.timeframe = newTimeframe;
        changeTimeframe(newTimeframe);
        updateKlineSubscription(); // Update WebSocket subscription on timeframe change
    }

    // Update the WebSocket subscription when timeframe changes
    function updateKlineSubscription() {
        if (state.ws && state.ws.readyState === WebSocket.OPEN) {
            const unsub = {
                "unsub": `market.${state.symbol.toLowerCase()}.kline.${state.timeframe}`
            };
            state.ws.send(JSON.stringify(unsub));

            const sub = {
                "sub": `market.${state.symbol.toLowerCase()}.kline.${state.timeframe}`,
                "id": `${state.symbol.toLowerCase()}_kline_${state.timeframe}`
            };
            state.ws.send(JSON.stringify(sub));
        }
    }

    // Initialize TradingView Chart on page load with default timeframe
    initTradingView(state.timeframe);

    // WebSocket Connection
    function connectWebSocket() {
        if (state.ws) {
            state.ws.close();
        }

        try {
            state.ws = new WebSocket('wss://api.huobi.pro/ws');

            state.ws.onopen = function() {
                console.log('WebSocket Connected');
                state.reconnectAttempts = 0;

                // Subscribe to market ticker
                const tickerSub = {
                    "sub": `market.${state.symbol.toLowerCase()}.ticker`,
                    "id": state.symbol
                };
                state.ws.send(JSON.stringify(tickerSub));

                // Subscribe to kline based on current timeframe
                const klineSub = {
                    "sub": `market.${state.symbol.toLowerCase()}.kline.${state.timeframe}`,
                    "id": `${state.symbol.toLowerCase()}_kline_${state.timeframe}`
                };
                state.ws.send(JSON.stringify(klineSub));
            };

            state.ws.onmessage = function(event) {
                try {
                    const blob = event.data;
                    if (typeof blob === 'string') {
                        handleWebSocketMessage(JSON.parse(blob));
                    } else {
                        // Handle binary message
                        const reader = new FileReader();
                        reader.onload = function() {
                            const data = JSON.parse(pako.inflate(reader.result, { to: 'string' }));
                            handleWebSocketMessage(data);
                        };
                        reader.readAsArrayBuffer(blob);
                    }
                } catch (e) {
                    console.warn('WebSocket message parsing error:', e);
                }
            };

            state.ws.onclose = function() {
                if (state.reconnectAttempts < state.maxReconnectAttempts) {
                    console.log(`WebSocket closed. Attempting to reconnect (${state.reconnectAttempts + 1}/${state.maxReconnectAttempts})`);
                    state.reconnectAttempts++;
                    const delay = Math.min(1000 * Math.pow(2, state.reconnectAttempts), 30000); // Exponential backoff with max 30s
                    setTimeout(connectWebSocket, delay);
                } else {
                    console.error('WebSocket connection failed after maximum attempts');
                    startPricePolling();
                }
            };

            state.ws.onerror = function(error) {
                console.warn('WebSocket error, will attempt to reconnect');
                state.ws.close();
            };

        } catch (error) {
            console.error('WebSocket connection error:', error);
            startPricePolling();
        }
    }

    // Handle WebSocket Messages
    function handleWebSocketMessage(data) {
        if (data.ping) {
            state.ws.send(JSON.stringify({ pong: data.ping }));
            return;
        }

        if (data.tick) {
            updatePriceDisplay(data.tick);
        }

        if (data.ch && data.ch.includes('kline')) {
            updateKlineDisplay(data.tick);
        }
    }

    // Update Price Display
    function updatePriceDisplay(tick) {
        state.currentPrice = tick.close;
        
        elements.currentPrice.textContent = `$${formatNumber(tick.close)}`;
        elements.mainPrice.textContent = `$${formatNumber(tick.close)}`;
        elements.volume.textContent = formatNumber(tick.vol);
        elements.high.textContent = `$${formatNumber(tick.high)}`;
        elements.low.textContent = `$${formatNumber(tick.low)}`;

        const change = ((tick.close - tick.open) / tick.open) * 100;
        const changeText = `${change >= 0 ? '+' : ''}${change.toFixed(2)}%`;
        elements.priceChange.innerHTML = `<span class="${change >= 0 ? 'text-green-500' : 'text-red-500'}">${changeText}</span>`;
        
        // Update main change display
        elements.mainChange.textContent = changeText;
        elements.mainChange.className = `${change >= 0 ? 'text-green-500' : 'text-red-500'} text-sm px-2 py-0.5 rounded-md ${change >= 0 ? 'bg-green-500/10' : 'bg-red-500/10'}`;

        updatePotentialReturn();
    }

    // Update Kline Display (if needed)
    function updateKlineDisplay(tick) {
        // You can process kline data here if needed
        // For example, updating additional UI elements based on kline data
        // Currently, TradingView widget handles chart updates
    }

    // Polling Fallback
    let pollingInterval;

    function startPricePolling() {
        if (pollingInterval) clearInterval(pollingInterval);
        pollingInterval = setInterval(async () => {
            try {
                const response = await fetch(`/trading/market-data?symbol=${state.symbol.toLowerCase()}`);
                if (response.ok) {
                    const data = await response.json();
                    if (data.success) {
                        updatePriceDisplay(data.data);
                    }
                }
            } catch (error) {
                console.warn('Price polling error:', error);
            }
        }, 3000);
    }

    // Event Handlers
    elements.riseButton.addEventListener('click', () => setTradeType('Rise'));
    elements.fallButton.addEventListener('click', () => setTradeType('Fall'));

    elements.tradeAmount.addEventListener('input', function() {
        state.amount = parseFloat(this.value) || 0;
        validateTrade();
        updatePotentialReturn();
    });

    document.querySelectorAll('.quick-amount').forEach(button => {
        button.addEventListener('click', () => {
            const amount = parseFloat(button.dataset.amount);
            if (!isNaN(amount)) {
                elements.tradeAmount.value = amount;
                state.amount = amount;
                validateTrade();
                updatePotentialReturn();
            }
        });
    });

    document.querySelectorAll('.duration-btn').forEach(button => {
        button.addEventListener('click', () => {
            document.querySelectorAll('.duration-btn').forEach(btn => {
                btn.classList.remove('border-[#2FE6DE]', 'text-[#2FE6DE]');
            });
            button.classList.add('border-[#2FE6DE]', 'text-[#2FE6DE]');
            state.duration = button.dataset.duration;
            validateTrade();
        });
    });

    document.getElementById('maxAmount').addEventListener('click', () => {
        const balance = parseFloat(elements.walletBalance.textContent.replace(/[^0-9.-]+/g, ''));
        if (!isNaN(balance)) {
            elements.tradeAmount.value = balance;
            state.amount = balance;
            validateTrade();
            updatePotentialReturn();
        }
    });

    elements.placeTrade.addEventListener('click', placeTrade);

    // Timeframe Buttons Event Listeners
    elements.timeframeButtons.forEach(button => {
        button.addEventListener('click', () => {
            const newTimeframe = button.dataset.timeframe;
            if (newTimeframe && newTimeframe !== state.timeframe) {
                handleTimeframeChange(newTimeframe);
            }
        });
    });

    // Trade Type Selection
    function setTradeType(type) {
        state.tradeType = type;
        
        // Update UI
        elements.riseButton.className = `p-4 rounded-lg text-center transition-all ${
            type === 'Rise' 
                ? 'bg-green-500/10 text-green-500 border-2 border-green-500' 
                : 'bg-[#0F0A1F]/50 text-gray-400 border-2 border-transparent hover:border-green-500/30 group'
        }`;
        
        elements.fallButton.className = `p-4 rounded-lg text-center transition-all ${
            type === 'Fall' 
                ? 'bg-red-500/10 text-red-500 border-2 border-red-500' 
                : 'bg-[#0F0A1F]/50 text-gray-400 border-2 border-transparent hover:border-red-500/30 group'
        }`;

        validateTrade();
    }

    // Update Potential Return
    function updatePotentialReturn() {
        if (state.amount > 0) {
            const potentialReturn = state.amount * (1 + state.profitPercentage / 100);
            elements.potentialReturn.innerHTML = `Potential Return: <span class="text-white">$${formatNumber(potentialReturn)}</span>`;
        } else {
            elements.potentialReturn.innerHTML = `Potential Return: <span class="text-white">$0.00</span>`;
        }
    }

    // Validate Trade
    function validateTrade() {
        const balance = parseFloat(elements.walletBalance.textContent.replace(/[^0-9.-]+/g, ''));
        const valid = state.tradeType && state.amount > 0 && state.amount <= balance;
        
        elements.placeTrade.disabled = !valid;
        elements.tradeButtonText.textContent = valid ? `Place ${state.tradeType} Trade` : 'Select Trade Type';
        
        if (state.amount > balance) {
            showError('Insufficient balance');
        } else {
            hideError();
        }
    }

    // Place Trade
    async function placeTrade() {
        if (elements.placeTrade.disabled) return;

        try {
            elements.placeTrade.disabled = true;
            elements.tradeButtonText.classList.add('hidden');
            elements.tradeButtonLoader.classList.remove('hidden');

            const csrfToken = document.querySelector('meta[name="csrf-token"]')?.content;
            if (!csrfToken) {
                throw new Error('CSRF token not found');
            }

            const response = await fetch('/user/trading/place-trade', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': csrfToken
                },
                body: JSON.stringify({
                    type: state.tradeType,
                    amount: state.amount,
                    duration: state.duration,
                    entry_price: state.currentPrice,
                    pair: state.symbol,
                    profit_percentage: state.profitPercentage
                })
            });

            const data = await response.json();

            if (data.success) {
                showSuccess('Trade placed successfully');
                elements.walletBalance.textContent = `$${formatNumber(data.new_balance)}`;
                elements.tradeAmount.value = '';
                state.amount = 0;
                updatePotentialReturn();
            } else {
                throw new Error(data.message || 'Failed to place trade');
            }

        } catch (error) {
            showError(error.message);
        } finally {
            elements.placeTrade.disabled = false;
            elements.tradeButtonLoader.classList.add('hidden');
            elements.tradeButtonText.classList.remove('hidden');
            validateTrade();
        }
    }

    // Error Handling
    function showError(message) {
        elements.amountError.textContent = message;
        elements.amountError.classList.remove('hidden');
    }

    function hideError() {
        elements.amountError.classList.add('hidden');
    }

    // Success Message
    function showSuccess(message) {
    // Using SweetAlert2 for success messages
    Swal.fire({
        title: 'Success',
        text: message,
        icon: 'success',
        background: '#0F0A1F',
        color: 'white',
        confirmButtonColor: '#2FE6DE',
        timer: 3000,
        timerProgressBar: true
    }).then((result) => {
        if (result.isConfirmed) {
            location.reload();
        }
    });
    }

    // Utility Functions
    function formatNumber(num) {
        if (isNaN(num)) return '0.00';
        return new Intl.NumberFormat('en-US', {
            minimumFractionDigits: 2,
            maximumFractionDigits: 2
        }).format(num);
    }

    // Error Modal Functions
    function showErrorModal(message) {
        document.getElementById('errorMessage').textContent = message;
        document.getElementById('errorModal').classList.remove('hidden');
    }

    function closeErrorModal() {
        document.getElementById('errorModal').classList.add('hidden');
    }

    // Initialize WebSocket Connection
    connectWebSocket();

    // Cleanup on page unload
    window.addEventListener('beforeunload', () => {
        if (state.ws) state.ws.close();
        if (pollingInterval) clearInterval(pollingInterval);
    });
});

// Demo mode toggle functionality
function toggleDemoMode(isDemo) {
    fetch('/user/trading/toggle-demo-mode', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
        },
        body: JSON.stringify({
            demo_mode: isDemo
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            location.reload(); // Refresh to update the page with new mode
        } else {
            alert('Failed to toggle demo mode');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Error toggling demo mode');
    });
}
</script>
        </main>
   
   
   
   <?php require ("footer.php") ?>